/*
 * InputConfig.h
 *
 * Created 8/28/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/28/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef INPUT_CONFIG_H
 #define INPUT_CONFIG_H
 
 template <typename T> class InputConfig;
 
 #include "Ralph.h" // should be changed to PlayableCharacter
 
 #include "genericAsyncTask.h"
 #include "asyncTaskManager.h"
 
 #include "global.h"
 #include "CameraAdjustment.h"
 #include "Vector.h"
 #include "Math3D.h"
 #include "referenceCount.h"
 #include "pointerTo.h"
 
 // Rotation speed measured in degrees
 #define ROTATION_SPEED 1.55f
 
 /**
  * Class specification for InputConfig
  */
 template <typename T>
 class InputConfig : public ReferenceCount
 {
 public:
    static enum ControlMode { THIRD_PERSON=0x0, FIRST_PERSON=0x1 };
 
 // Data Members
 private:
    PT(Ralph<T>) _player_Ptr; // the player's character
    NodePath _cam; // the camera for the player's character
    
    bool _move_forward;
    bool _move_backward;
    bool _move_left;
    bool _move_right;
    bool _rotate_left;
    bool _rotate_right;
    
    bool _prioritize_move_forward; // if false, prioritize move backward
    bool _prioritize_move_left; // if false, prioritize move right
    bool _prioritize_rotate_left; // if false, prioritize rotate right
    
    void (Ralph< PRIMARY_TYPE >::*_move_forward_function_Ptr)(); // move (e.g. run or walk) function pointer
    
    std::string _close_window_key;
    std::string _move_forward_key;
    std::string _move_backward_key;
    std::string _move_left_key;
    std::string _move_right_key;
    std::string _rotate_left_key;
    std::string _rotate_right_key;
    
    std::string _walk_key;
    bool _walk_key_on;
    
    std::string _standard_attack_key;
    std::string _projectile_attack_key;
    std::string _dash_key;
    std::string _adjust_cam_to_player_key;
    
    ControlMode _control_mode;
    T _third_person_camera_angle; // measured in degrees
 
 // Local Functions
 public:
    InputConfig( Ralph<T>* player_Ptr,
                 NodePath& cam,
                 const std::string& move_forward_key = "w", 
                 const std::string& move_backward_key = "s",
                 const std::string& move_left_key = "a",
                 const std::string& move_right_key = "d",
                 const std::string& rotate_left_key = "q",
                 const std::string& rotate_right_key = "e",
                 const std::string& adjust_cam_to_player_key = "r",
                 const std::string& walk_key = "l", //"shift"
                 const std::string& standard_attack_key = "j",
                 const std::string& projectile_attack_key = "i",
                 const std::string& dash_key = "space" );
    InputConfig( const InputConfig<T>& input_config );
    virtual ~InputConfig();
    inline InputConfig<T>& operator=( const InputConfig<T>& input_config );
    inline void bind_keyboard_inputs( WindowFramework& window );
    inline NodePath& get_camera();
    inline Ralph<T>* InputConfig<T>::get_character();
    inline VECTOR2_TYPE get_unit_xy_movement_direction( const T& degrees );
    inline VECTOR2_TYPE get_unit_xy_movement_direction( const VECTOR2_TYPE& dir );
    inline VECTOR2_TYPE get_unit_xy_movement_direction_standard( const T& degrees );
    inline VECTOR2_TYPE get_unit_xy_movement_direction_standard( const VECTOR2_TYPE& dir );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    static inline AsyncTask::DoneStatus adjust_camera_heading_gradually( GenericAsyncTask* task_Ptr, void* data_Ptr );
    static inline AsyncTask::DoneStatus apply_input_controls( GenericAsyncTask* task_Ptr, void* data_Ptr );
    static inline void apply_first_person_controls( InputConfig<T>& input_config, const T& fps_ratio_offset );
    static inline void apply_third_person_controls( InputConfig<T>& input_config, const T& fps_ratio_offset );
    static inline void apply_movement( Ralph<T>& character, const VECTOR2_TYPE& displacement );
    static inline void apply_rotation( InputConfig<T>& input_config, NodePath& node_path, const T& rotation_speed );
    static inline void set_walk_key( const Event * event_Ptr, void * data_Ptr );
    static inline void reset_walk_key( const Event * event_Ptr, void * data_Ptr );
    //static inline void exitProgram( const Event * event_Ptr, void * data_Ptr );
    static inline void start_move_forward( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_move_forward( const Event * event_Ptr, void * data_Ptr );
    static inline void start_move_backward( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_move_backward( const Event * event_Ptr, void * data_Ptr );
    static inline void start_move_left( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_move_left( const Event * event_Ptr, void * data_Ptr );
    static inline void start_move_right( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_move_right( const Event * event_Ptr, void * data_Ptr );
    static inline void start_rotate_left( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_rotate_left( const Event * event_Ptr, void * data_Ptr );
    static inline void start_rotate_right( const Event * event_Ptr, void * data_Ptr );
    static inline void stop_rotate_right( const Event * event_Ptr, void * data_Ptr );
    static inline void adjust_camera_heading( const Event * event_Ptr, void * data_Ptr );
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 InputConfig<T>::InputConfig( Ralph<T>* player_Ptr,
                              NodePath& cam,
                              const std::string& move_forward_key, 
                              const std::string& move_backward_key,
                              const std::string& move_left_key,
                              const std::string& move_right_key,
                              const std::string& rotate_left_key,
                              const std::string& rotate_right_key,
                              const std::string& adjust_cam_to_player_key,
                              const std::string& walk_key,
                              const std::string& standard_attack_key,
                              const std::string& projectile_attack_key,
                              const std::string& dash_key )
                : _player_Ptr( player_Ptr ),
                  _cam( cam ),
                  _move_forward( false ),
                  _move_backward( false ),
                  _move_left( false ),
                  _move_right( false ),
                  _rotate_left( false ),
                  _rotate_right( false ),
                  _prioritize_move_forward( true ), // if false, prioritize move backward
                  _prioritize_move_left( true ), // if false, prioritize move right
                  _prioritize_rotate_left( true ), // if false, prioritize rotate right
                  _move_forward_function_Ptr( &Ralph<T>::run_forward ),
                  _close_window_key( "escape" ),
                  _move_forward_key( move_forward_key ),
                  _move_backward_key( move_backward_key ),
                  _move_left_key( move_left_key ),
                  _move_right_key( move_right_key ),
                  _rotate_left_key( rotate_left_key ),
                  _rotate_right_key( rotate_right_key ),
                  _adjust_cam_to_player_key( adjust_cam_to_player_key ),
                  _walk_key( walk_key ),
                  _walk_key_on( false ),
                  _standard_attack_key( standard_attack_key ),
                  _projectile_attack_key( projectile_attack_key ),
                  _dash_key( dash_key ),
                  _control_mode( THIRD_PERSON ),
                  _third_person_camera_angle( ZERO )
 {
    nassertv( _player_Ptr != NULL );
    nassertv( !_cam.is_empty() );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 InputConfig<T>::InputConfig( const InputConfig<T>& input_config )
                : _player_Ptr( input_config._player_Ptr ),
                  _cam( input_config._cam ),
                  _move_forward( input_config._move_forward ),
                  _move_backward( input_config._move_backward ),
                  _move_left( input_config._move_left ),
                  _move_right( input_config._move_right ),
                  _rotate_left( input_config._rotate_left ),
                  _rotate_right( input_config._rotate_right ),
                  _prioritize_move_forward( input_config._prioritize_move_forward ),
                  _prioritize_move_left( input_config._prioritize_move_left ),
                  _prioritize_rotate_left( input_config._prioritize_rotate_left ),
                  _move_forward_function_Ptr( input_config._move_forward_function_Ptr ),
                  _close_window_key( input_config._close_window_key ),
                  _move_forward_key( input_config._move_forward_key ),
                  _move_backward_key( input_config._move_backward_key ),
                  _move_left_key( input_config._move_left_key ),
                  _move_right_key( input_config._move_right_key ),
                  _rotate_left_key( input_config._rotate_left_key ),
                  _rotate_right_key( input_config._rotate_right_key ),
                  _adjust_cam_to_player_key( input_config._adjust_cam_to_player_key ),
                  _walk_key( input_config._walk_key ),
                  _walk_key_on( input_config._walk_key_on ),
                  _standard_attack_key( input_config._standard_attack_key ),
                  _projectile_attack_key( input_config._projectile_attack_key ),
                  _dash_key( input_config._dash_key ),
                  _control_mode( input_config._control_mode ),
                  _third_person_camera_angle( input_config._third_person_camera_angle )
 {
 
 }
 
 /**
  * Destructor
  */
 template <typename T>
 InputConfig<T>::~InputConfig()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified InputConfig to this InputConfig.
  *
  * @param (const InputConfig<T>& input_config )
  * @return InputConfig<T>&
  */
 template <typename T>
 inline InputConfig<T>& InputConfig<T>::operator=( const InputConfig<T>& input_config )
 {
    _player_Ptr = input_config._player_Ptr;
    _cam = input_config._cam;
    _move_forward = input_config._move_forward;
    _move_backward = input_config._move_backward;
    _move_left = input_config._move_left;
    _move_right = input_config._move_right;
    _rotate_left = input_config._rotate_left;
    _rotate_right = input_config._rotate_right;
    _prioritize_move_forward = input_config._prioritize_move_forward;
    _prioritize_move_left = input_config._prioritize_move_left;
    _prioritize_rotate_left = input_config._prioritize_rotate_left;
    _move_forward_function_Ptr = input_config._move_forward_function_Ptr;
    _close_window_key = input_config._close_window_key;
    _move_forward_key = input_config._move_forward_key;
    _move_backward_key = input_config._move_backward_key;
    _move_left_key = input_config._move_left_key;
    _move_right_key = input_config._move_right_key;
    _rotate_left_key = input_config._rotate_left_key;
    _rotate_right_key = input_config._rotate_right_key;
    _adjust_cam_to_player_key = input_config._adjust_cam_to_player_key;
    _walk_key = input_config._walk_key;
    _walk_key_on = input_config._walk_key_on;
    _standard_attack_key = input_config._standard_attack_key;
    _projectile_attack_key = input_config._projectile_attack_key;
    _dash_key = input_config._dash_key;
    _control_mode = input_config._control_mode;
    _third_person_camera_angle = input_config._third_person_camera_angle;
    
    return *this;
 }
 
 /**
  * bind_keyboard_inputs() binds the appropriate keys for the keyboard controls for manipulating
  * the player's character.
  *
  * @param (WindowFramework&) window
  */
 template <typename T>
 inline void InputConfig<T>::bind_keyboard_inputs( WindowFramework& window )
 {
    // Enable Keyboard Detection
    window.enable_keyboard();
    
    global::_framework.define_key( _close_window_key, "Close window", &PandaFramework::event_esc, static_cast<void*>(&global::_framework) );
    //global::_framework.define_key( "escape", "Close window", &exitProgram, (void*) NULL );
    global::_framework.define_key( _move_forward_key, "Move forward", &InputConfig<T>::start_move_forward, static_cast<void*>(this) );
    global::_framework.define_key( _move_forward_key + "-up", "Stop moving forward", &InputConfig<T>::stop_move_forward, static_cast<void*>(this) );
    global::_framework.define_key( _move_backward_key, "Move backward", &InputConfig<T>::start_move_backward, static_cast<void*>(this) );
    global::_framework.define_key( _move_backward_key + "-up", "Stop moving backward", &InputConfig<T>::stop_move_backward, static_cast<void*>(this) );
    global::_framework.define_key( _move_left_key, "Move left", &InputConfig<T>::start_move_left, static_cast<void*>(this) );
    global::_framework.define_key( _move_left_key + "-up", "Stop moving left", &InputConfig<T>::stop_move_left, static_cast<void*>(this) );
    global::_framework.define_key( _move_right_key, "Move right", &InputConfig<T>::start_move_right, static_cast<void*>(this) );
    global::_framework.define_key( _move_right_key + "-up", "Stop moving right", &InputConfig<T>::stop_move_right, static_cast<void*>(this) );
    global::_framework.define_key( _rotate_left_key, "Rotate left", &InputConfig<T>::start_rotate_left, static_cast<void*>(this) );
    global::_framework.define_key( _rotate_left_key + "-up", "Stop rotating left", &InputConfig<T>::stop_rotate_left, static_cast<void*>(this) );
    global::_framework.define_key( _rotate_right_key, "Rotate right", &InputConfig<T>::start_rotate_right, static_cast<void*>(this) );
    global::_framework.define_key( _rotate_right_key + "-up", "Stop rotating right", &InputConfig<T>::stop_rotate_right, static_cast<void*>(this) );
    global::_framework.define_key( _adjust_cam_to_player_key, "Sets the camera to face in the direction of the character", &InputConfig<T>::adjust_camera_heading, static_cast<void*>(this) );
    global::_framework.define_key( _walk_key, "Start walking", &InputConfig<T>::set_walk_key, static_cast<void*>(this) );
    global::_framework.define_key( _walk_key + "-up", "Start running", &InputConfig<T>::reset_walk_key, static_cast<void*>(this) );
    global::_framework.define_key( _standard_attack_key, "Standard attack", &Ralph<T>::execute_standard_attack, static_cast<void*>(this->_player_Ptr) );
    global::_framework.define_key( _projectile_attack_key, "Projectile attack", &Ralph<T>::execute_projectile_attack, static_cast<void*>(this->_player_Ptr) );
    global::_framework.define_key( _dash_key, "Dash", &Ralph<T>::dash, static_cast<void*>(this->_player_Ptr) );
    
    //printf( "%1d\n", EventHandler::get_global_event_handler()->has_hook( "shift-a" ) );
    
    //global::_task_mgr_Ptr->add( new GenericAsyncTask( "Applies Input Controls", InputConfig<T>::apply_input_controls, static_cast<void*>(character_Ptr) ) );
    global::_task_mgr_Ptr->add( new GenericAsyncTask( "Input Controls for " + _cam.get_key(), InputConfig<T>::apply_input_controls, static_cast<void*>(this) ) );
 }
 
 /**
  * get_camera() returns the camera of this InputConfig.
  *
  * @return NodePath&
  */
 template <typename T>
 inline NodePath& InputConfig<T>::get_camera()
 {
    return _cam;
 }
 
 /**
  * get_character() returns the character of this InputConfig.
  *
  * @return Ralph<T>*
  */
 template <typename T>
 inline Ralph<T>* InputConfig<T>::get_character()
 {
    return _player_Ptr;
 }
 
 /**
  * get_unit_xy_movement_direction() returns the unit vector representing the direction
  * of the user's input movement on the xy-plane based on the specified angle the 
  * actor is facing. The geographical coordinate system is assumed.
  *
  * @param (const T&) degrees
  * @return VECTOR2_TYPE
  */
 template <typename T>
 inline VECTOR2_TYPE InputConfig<T>::get_unit_xy_movement_direction( const T& degrees )
 {
    return InputConfig<T>::get_unit_xy_movement_direction( Vector::get_xy_direction( degrees ) );
 }
 
 /**
  * get_unit_xy_movement_direction() returns the unit vector representing the direction
  * of the user's input movement on the xy-plane based on the specified direction the 
  * actor is facing. The geographical coordinate system is assumed.
  *
  * @param (const VECTOR2_TYPE&) dir
  * @return VECTOR2_TYPE
  */
 template <typename T>
 inline VECTOR2_TYPE InputConfig<T>::get_unit_xy_movement_direction( const VECTOR2_TYPE& dir )
 {
    PRIMARY_TYPE displacement_x( ZERO );
    PRIMARY_TYPE displacement_y( ZERO );
    
    if ( InputConfig<T>::_move_forward  )
    {
        if ( !InputConfig<T>::_move_backward || InputConfig<T>::_prioritize_move_forward )
        {
            displacement_x += dir.get_x();
            displacement_y += dir.get_y();
        }
        else // InputConfig::_move_backward && !InputConfig::_prioritize_move_forward
        {
            displacement_x -= dir.get_x();
            displacement_y -= dir.get_y();
        }
    }
    else if ( InputConfig<T>::_move_backward && !InputConfig<T>::_move_forward )
    {
        displacement_x -= dir.get_x();
        displacement_y -= dir.get_y();
    }
    
    // Geographical coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 0, 1), (-dir_x, dir_y, 0) )
    // = (-dir_y, dir_x, 0 )
    // PRIMARY_TYPE temp( dir_x );
    // dir_x = -dir_y;
    // dir_y = temp;
    
    // Standard xyz-coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 1, 0), (dir_x, 0, dir_z) )
    // = (dir_z, 0, -dir_x)
    // PRIMARY_TYPE temp( -dir_x );
    // dir_x = dir_z;
    // dir_z = temp;
    
    if ( InputConfig<T>::_move_left )
    {
        if ( !InputConfig<T>::_move_right || InputConfig<T>::_prioritize_move_left )
        {
            //displacement_x += dir_x;
            //displacement_y += dir_y;
            displacement_x -= dir.get_y();
            displacement_y += dir.get_x();
        }
        else // InputConfig::_move_right && !InputConfig::_prioritize_move_left
        {
            //displacement_x -= dir_x;
            //displacement_y -= dir_y;
            displacement_x += dir.get_y();
            displacement_y -= dir.get_x();
        }
    }
    else if ( InputConfig<T>::_move_right && !InputConfig<T>::_move_left )
    {
        //displacement_x -= dir_x;
        //displacement_y -= dir_y;
        displacement_x += dir.get_y();
        displacement_y -= dir.get_x();
    }
    
    // normalize the displacements
    if ( displacement_x != ZERO || displacement_y != ZERO )
        Vector::normalize( displacement_x, displacement_y );
    
    return VECTOR2_TYPE( displacement_x, displacement_y );
 }
 
 /**
  * get_unit_xy_movement_direction_standard() returns the unit vector representing the direction
  * of the user's input movement on the xz-plane based on the specified angle the actor is facing. 
  * (albeit, we generalized the function to also work for a 2-dimensional world, hence, xy is in 
  * the function's name). The standard xyz-coordinate system is assumed.
  *
  * @param (const T&) degrees
  * @return VECTOR2_TYPE
  */
 template <typename T>
 inline VECTOR2_TYPE InputConfig<T>::get_unit_xy_movement_direction_standard( const T& degrees )
 {
    return InputConfig<T>::get_unit_xy_movement_direction_standard( Vector::get_xy_direction_standard( degrees ) );
 }
 
 /**
  * get_unit_xy_movement_direction_standard() returns the unit vector representing the direction
  * of the user's input movement on the xz-plane based on the specified direction the actor is facing. 
  * (albeit, we generalized the function to also work for a 2-dimensional world, hence, xy is in 
  * the function's name). The standard xyz-coordinate system is assumed.
  *
  * @param (const VECTOR2_TYPE&) dir
  * @return VECTOR2_TYPE
  */
 template <typename T>
 inline VECTOR2_TYPE InputConfig<T>::get_unit_xy_movement_direction_standard( const VECTOR2_TYPE& dir )
 {
    PRIMARY_TYPE displacement_x( ZERO );
    PRIMARY_TYPE displacement_y( ZERO );
    
    if ( InputConfig<T>::_move_forward  )
    {
        if ( !InputConfig<T>::_move_backward || InputConfig<T>::_prioritize_move_forward )
        {
            displacement_x += dir.get_x();
            displacement_y += dir.get_y();
        }
        else // InputConfig::_move_backward && !InputConfig::_prioritize_move_forward
        {
            displacement_x -= dir.get_x();
            displacement_y -= dir.get_y();
        }
    }
    else if ( InputConfig<T>::_move_backward && !InputConfig<T>::_move_forward )
    {
        displacement_x -= dir.get_x();
        displacement_y -= dir.get_y();
    }
    
    // Geographical coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 0, 1), (-dir_x, dir_y, 0) )
    // = (-dir_y, dir_x, 0 )
    // PRIMARY_TYPE temp( dir_x );
    // dir_x = -dir_y;
    // dir_y = temp;
    
    // Standard xyz-coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 1, 0), (dir_x, 0, dir_z) )
    // = (dir_z, 0, -dir_x)
    // PRIMARY_TYPE temp( -dir_x );
    // dir_x = dir_z;
    // dir_z = temp;
    
    if ( InputConfig<T>::_move_left )
    {
        if ( !InputConfig<T>::_move_right || InputConfig<T>::_prioritize_move_left )
        {
            //displacement_x += dir_x;
            //displacement_y += dir_y;
            displacement_x += dir.get_y();
            displacement_y -= dir.get_x();
        }
        else // InputConfig::_move_right && !InputConfig::_prioritize_move_left
        {
            //displacement_x -= dir_x;
            //displacement_y -= dir_y;
            displacement_x -= dir.get_y();
            displacement_y += dir.get_x();
        }
    }
    else if ( InputConfig<T>::_move_right && !InputConfig<T>::_move_left )
    {
        //displacement_x -= dir_x;
        //displacement_y -= dir_y;
        displacement_x -= dir.get_y();
        displacement_y += dir.get_x();
    }
    
    // normalize the displacements
    if ( displacement_x != ZERO || displacement_y != ZERO )
        Vector::normalize( displacement_x, displacement_y );
    
    return VECTOR2_TYPE( displacement_x, displacement_y );
 }
 
 /** STATIC FUNCTIONS **/
 
 /**
  * adjust_camera_heading_gradually() sets the camera heading angle of the specified InputConfig 
  * to face in the direction of the corresponding character gradually, but at a quick pace.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus InputConfig<T>::adjust_camera_heading_gradually( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    nassertr( data_Ptr != NULL, AsyncTask::DS_done );
    
    PT(InputConfig<T>) input_config_Ptr( static_cast<InputConfig<T>*>( data_Ptr ) );
    
    CameraAdjustment::adjust_camera_heading_gradually( input_config_Ptr->_cam, input_config_Ptr->_player_Ptr->get_h(), 0.25f, 15.0f );
    
    if ( input_config_Ptr->_cam.get_h() == input_config_Ptr->_player_Ptr->get_h() )
        return AsyncTask::DS_done;
    else
        return AsyncTask::DS_cont;
 }
 
 /**
  * apply_input_controls() applies the input controls every frame.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus InputConfig<T>::apply_input_controls( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {  
    nassertr( data_Ptr != NULL, AsyncTask::DS_done );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    nassertr( input_config_Ptr->_player_Ptr != NULL, AsyncTask::DS_done );
    nassertr( !input_config_Ptr->_cam.is_empty(), AsyncTask::DS_done );
    Ralph<T>& character( *input_config_Ptr->get_character() );
    NodePath& cam( input_config_Ptr->get_camera() );
    
    // if the player is can move
    if ( character.is_mobile() )
    {
        //global::_clock_Ptr->tick();
        //printf( "%1f\n", global::_clock_Ptr->get_frame_time() );
        PRIMARY_TYPE fps( global::_clock_Ptr->get_average_frame_rate() );
        PRIMARY_TYPE fps_ratio_offset( fps == ZERO ? ONE : BASE_FPS / fps );
        
        if ( input_config_Ptr->_control_mode == InputConfig<T>::FIRST_PERSON )
            InputConfig<T>::apply_first_person_controls( *input_config_Ptr, fps_ratio_offset );
        else // InputConfig::_control_mode == InputConfig::THIRD_PERSON
            InputConfig<T>::apply_third_person_controls( *input_config_Ptr, fps_ratio_offset );
    }
    else // the player cannot move
    {
        if ( input_config_Ptr->_control_mode == InputConfig<T>::FIRST_PERSON )
        {
            // Apply rotation
            InputConfig<T>::apply_rotation( *input_config_Ptr, character, ROTATION_SPEED );
            
            // Readjust camera
            CameraAdjustment::adjust_camera_to_first_person_character( cam, character, CAMERA_XY_DISTANCE );
        }
        else // input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON
        {
            // Apply rotation
            InputConfig<T>::apply_rotation( *input_config_Ptr, cam, ROTATION_SPEED );
            
            // Move the camera to a location that is the specified distance away from the character 
            // without changing the heading angle of the camera, and such that the camera is directly 
            // facing the character
            CameraAdjustment::adjust_camera_to_third_person_character( cam, character, CAMERA_XY_DISTANCE );
        }
    }
    
    return AsyncTask::DS_cont;
 }
 
 /**
  * apply_first_person_controls() applies first-person-like controls to the inputs 
  * for the character of the specified InputConfig.
  * The controlled character will still be viewed from the third-person perspective. 
  * Do not confuse this with a first-person camera view.
  *
  * @param (InputConfig<T>&) input_config
  * @param (const T&) fps_ratio_offset - affects the movement rate of the inputs
  */
 template <typename T>
 inline void InputConfig<T>::apply_first_person_controls( InputConfig<T>& input_config, const T& fps_ratio_offset )
 {
    nassertv( input_config._player_Ptr != NULL );
    nassertv( !input_config._cam.is_empty() );
    Ralph<T>& character( *input_config.get_character() );
    NodePath& cam( input_config.get_camera() );
    
    // displacement is already normalized (multiplied by the fps_ratio_offset)
    VECTOR2_TYPE displacement( input_config.get_unit_xy_movement_direction( character.get_h() ) * fps_ratio_offset );
    
    // Apply movement
    InputConfig<T>::apply_movement( character, displacement );
    
    // Apply rotation
    InputConfig<T>::apply_rotation( input_config, character, ROTATION_SPEED );
    
    // Apply animation
    if ( displacement.get_x() != ZERO || displacement.get_y() != ZERO )
    {
        (character.*input_config._move_forward_function_Ptr)();
    }
    else
    {
        character.assume_neutral_position();
    }
    
    // Readjust camera
    CameraAdjustment::adjust_camera_to_first_person_character( cam, character, CAMERA_XY_DISTANCE );
 }
 
 /**
  * apply_third_person_controls() applies the standard third-person controls to the inputs
  * for the character of the specified InputConfig.
  * It is coincident that the controlled character is viewed from the third-person
  * perspective. This function has nothing to do with the third-person camera view; this 
  * strictly applies to the player's controls.
  *
  * @param (InputConfig<T>&) input_config
  * @param (const T&) fps_ratio_offset - affects the movement rate of the inputs
  */
 template <typename T>
 inline void InputConfig<T>::apply_third_person_controls( InputConfig<T>& input_config, const T& fps_ratio_offset )
 {
    nassertv( input_config._player_Ptr != NULL );
    nassertv( !input_config._cam.is_empty() );
    Ralph<T>& character( *input_config.get_character() );
    NodePath& cam( input_config.get_camera() );
    
    // displacement is a unit vector (multiplied by fps_ratio_offset)
    VECTOR2_TYPE displacement( input_config.get_unit_xy_movement_direction( input_config._third_person_camera_angle ) * fps_ratio_offset );
    
    // Apply movement
    InputConfig<T>::apply_movement( character, displacement );
    
    // Apply rotation
    InputConfig<T>::apply_rotation( input_config, cam, ROTATION_SPEED );
    
    // Readjust camera and apply animation
    if ( displacement.get_x() != ZERO || displacement.get_y() != ZERO )
    {
        // Slowly readjust/spin the camera to be in back of the controlled character
        CameraAdjustment::adjust_camera_heading_gradually( cam, character.get_h(), 
             THIRD_PERSON_VIEW_READJUSTMENT_RATIO * fps_ratio_offset, THIRD_PERSON_VIEW_MIN_READJUSTMENT * fps_ratio_offset );
        
        character.look_at_xy_direction( displacement.get_x(), displacement.get_y() );
        
        (character.*input_config._move_forward_function_Ptr)();
    }
    else
    {
        character.assume_neutral_position();
    }
    
    // Move the camera to a location that is the specified distance away from the character 
    // without changing the heading angle of the camera, and such that the camera is directly 
    // facing the character
    CameraAdjustment::adjust_camera_to_third_person_character( cam, character, CAMERA_XY_DISTANCE );
 }
 
 /**
  * apply_movement() applies movement, based on the user's inputs and the specified 
  * movement direction offset by the frame-per-second ratio offset (fps_ratio_offset), 
  * to the specified character.
  *
  * @param (Ralph<T>&) character
  * @param (const VECTOR2_TYPE&) displacement
  */
 template <typename T>
 inline void InputConfig<T>::apply_movement( Ralph<T>& character, const VECTOR2_TYPE& displacement )
 {
    // if displacement has occurred, update position
    if ( displacement.get_x() != ZERO )
    {
        character.set_x( character.get_x() + (displacement.get_x() * character.get_movement_speed()) );
    }
    if ( displacement.get_y() != ZERO )
    {
        character.set_y( character.get_y() + (displacement.get_y() * character.get_movement_speed()) );
    }
 }
 
 /**
  * apply_rotation() applies rotation, based on the user's inputs and the specified 
  * rotation_speed, to the specified NodePath. The inputs are determined based on the 
  * specified InputConfig.
  *
  * @param (InputConfig<T>&) input_config
  * @param (NodePath&) node_path
  * @param (const T&) rotation_speed - measured in degrees (this should always be non-negative)
  */
 template <typename T>
 inline void InputConfig<T>::apply_rotation( InputConfig<T>& input_config, NodePath& node_path, const T& rotation_speed )
 {
    if ( input_config._rotate_left )
    {
        if ( !input_config._rotate_right || input_config._prioritize_rotate_left )
        {
            node_path.set_h( node_path.get_h() + rotation_speed );
        }
        else // input_config._rotate_right && !input_config._prioritize_rotate_left
        {
            node_path.set_h( node_path.get_h() - rotation_speed );
        }
    }
    else if ( input_config._rotate_right && !input_config._rotate_left)
    {
        node_path.set_h( node_path.get_h() - rotation_speed );
    }
 }
 
 /**
  * set_walk_key() turns the walk key on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::set_walk_key( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_walk_key_on = true;
    
    input_config_Ptr->_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::walk_forward;
 }
 
 /**
  * reset_walk_key() turns the walk key off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::reset_walk_key( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_walk_key_on = false;
    
    input_config_Ptr->_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::run_forward;
 }
 
 /**
  * exitProgram() exits the program.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 //inline void exitProgram( const Event * event_Ptr, void * data_Ptr )
 //{
    //exit( 0 );
 //}
 
 /**
  * start_move_forward() turns the _move_forward flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_move_forward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_forward = true;
    
    if ( !input_config_Ptr->_prioritize_move_forward )
        input_config_Ptr->_prioritize_move_forward = true;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * stop_move_forward() turns the _move_forward flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_move_forward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_forward = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * start_move_backward() turns the _move_backward flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_move_backward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_backward = true;
    
    if ( input_config_Ptr->_prioritize_move_forward )
        input_config_Ptr->_prioritize_move_forward = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * stop_move_backward() turns the _move_backward flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_move_backward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_backward = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * start_move_left() turns the _move_left flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_move_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_left = true;
    
    if ( !input_config_Ptr->_prioritize_move_left )
        input_config_Ptr->_prioritize_move_left = true;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * stop_move_left() turns the _move_left flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_move_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_left = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * start_move_right() turns the _move_right flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_move_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_right = true;
    
    if ( input_config_Ptr->_prioritize_move_left )
        input_config_Ptr->_prioritize_move_left = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * stop_move_right() turns the _move_right flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_move_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_move_right = false;
    
    if ( input_config_Ptr->_control_mode == InputConfig<T>::THIRD_PERSON )
        input_config_Ptr->_third_person_camera_angle = input_config_Ptr->_cam.get_h();
 }
 
 /**
  * start_rotate_left() turns the _rotate_left flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_rotate_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_rotate_left = true;
    
    if ( !input_config_Ptr->_prioritize_rotate_left )
        input_config_Ptr->_prioritize_rotate_left = true;
 }
 
 /**
  * stop_rotate_left() turns the _rotate_left flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_rotate_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    static_cast<InputConfig<T>*>(data_Ptr)->_rotate_left = false;
 }
 
 /**
  * start_rotate_right() turns the _rotate_right flag on.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::start_rotate_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    InputConfig<T>* input_config_Ptr( static_cast<InputConfig<T>*>(data_Ptr) );
    
    input_config_Ptr->_rotate_right = true;
    
    if ( input_config_Ptr->_prioritize_rotate_left )
        input_config_Ptr->_prioritize_rotate_left = false;
 }
 
 /**
  * stop_rotate_right() turns the _rotate_right flag off.
  * The data_Ptr is expected to be pointing to an InputConfig.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::stop_rotate_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    static_cast<InputConfig<T>*>(data_Ptr)->_rotate_right = false;
 }
 
 /**
  * adjust_camera_heading() sets the camera heading angle of the specified InputConfig 
  * to face in the direction of the corresponding character.
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void InputConfig<T>::adjust_camera_heading( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    
    global::_task_mgr_Ptr->add( new GenericAsyncTask( "Adjust Camera Heading Angle", InputConfig<T>::adjust_camera_heading_gradually, data_Ptr ) );
    
    //PT(InputConfig<T>) input_config_Ptr( static_cast<InputConfig<T>*>( data_Ptr ) );
    
    //input_config_Ptr->_cam.set_h( input_config_Ptr->_player_Ptr->get_h() );
 }
 
 #undef ROTATION_SPEED
 
 #endif // INPUT_CONFIG_H